/* daemon.c - a generic daemon. Possibly useful as a template */
/* Written for the Linux Operating System                     */
/* compile: cc -o daemond daemon.c dbugger.c sig_ign.c        */

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <termio.h>
#include <time.h>
#include <unistd.h>
#include "dbugger.h"


/* **************** GLOBAL VARIABLES ************************** */
#ifndef TRUE	
	#define TRUE 1
#endif
#ifndef FALSE	
	#define FALSE 0
#endif
#ifndef GOOD	
	#define GOOD 0
#endif
#ifndef BAD	
	#define BAD -1
#endif

int  Ready_Flag = TRUE;
int  Reset_Flag = FALSE;



/* ******* Forward Declarations *************** */
void main             ( int, char** );
void sigusr           ( int );
void init_daemon      ( void );
void the_daemon       ( void );
void process_message  ( char* );
int  msg_handler_zero ( char* );
void shutdown_daemon  ( void );


/* ********* Pipe globals and functions ********** */
void   open_pipe   ( char*, int );
int    verify_pipe ( char* );
int    read_pipe   ( char* );
void   close_pipe  ( void );
char   pipe_name[40];
int    Pipe_Message_Length = 80; /* seems reasonable */
int    Pipe_Handle;
#define FIFO_MODE (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP)


/* ******* Semaphore globals and functions ******** */
int    get_semaphore  ( int );
int    set_semaphore  ( int );
int    check_semaphore( int );
void   del_semaphore  ( int );
static int sem_id;
int    DAEMON_SEMA  = 4142;  /* The semaphore key */


/* ****** Required globals for dbugger() ****** */
int    DEBUG_LEVEL  = 9;  /* 0 - 9, nine catches all dbug msgs */
long   DEBUG_MAX    = 100000L;
char   DEBUG_FILE[40];
char  *program_name;


/* ****** Operation Integrity check ****** */
int  check_dirty_bit( void );
int set_bit_clean  ( void );
int set_bit_dirty  ( void );
int  system_dirty   ( void );
char INTEGRITY_FILE[40];




/* ********** Start of Code    ************* */

void main(int argc, char **argv)
{
struct sigaction sigact;

	if( ( program_name = strrchr( *argv, '/') ) == ( char * ) NULL )
		{ program_name = *argv; } else { program_name++; }

	sprintf( DEBUG_FILE,     "/tmp/%s.dbug", program_name );
	sprintf( INTEGRITY_FILE, "/tmp/%s.last", program_name );
	sprintf( pipe_name,      "/tmp/%s.fifo", program_name );
	argc = argc;

	del_semaphore( DAEMON_SEMA );

	if( check_semaphore( DAEMON_SEMA ) == GOOD )
	{
		fprintf(stderr,"Semaphore %d is already taken\n", DAEMON_SEMA);
		fprintf(stderr,"A copy of %s may already be running\n", program_name);
		fprintf(stderr,"Exiting...\n", program_name);
		exit(BAD);
		
	}
	fprintf(stderr,"Semaphore %d will be used\n", DAEMON_SEMA);

	if( check_dirty_bit() == TRUE) 
	{
		fprintf(stderr,
			"%s previously went down dirty, Trying to recover...\n",
			program_name);
		if( system_dirty() == TRUE )
		{
				fprintf(stderr,"%s Recovery Fails, Exiting...\n",program_name);
				exit(BAD);
		}
	}
	else
		fprintf(stderr,"Previous %s process terminated cleanly\n",program_name);

	fclose(stdin);
	fclose(stdout);
	fclose(stderr);
	setpgrp();

	sigact.sa_handler = SIG_IGN;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;
	if (sigaction(SIGCHLD, &sigact, (void *) NULL))
	{
			dbugger(MANDATORY, "Can't get set the initial sigaction\n\n");
			exit(BAD);
	}

	switch(fork())
	{
		case 0:
			dbugger(MANDATORY, "Program: %s is forked in the background", program_name);
			the_daemon();
			break;
		case -1:
			dbugger(MANDATORY, "Program: %s is NOT forked the background (Error Num: %d)",
				program_name, errno);
			exit(BAD);
			break;
	}

	exit(GOOD);
} /* end main */


void the_daemon(void)
{
char Message[Pipe_Message_Length];

	init_daemon();
	while (Ready_Flag)
	{
		if (read_pipe(Message))
			process_message(Message);

		if (Reset_Flag)
		{
			Reset_Flag = 0;
			shutdown_daemon();
			sleep(1);
			init_daemon();
		}
		usleep(100000);
	}
	shutdown_daemon();

} /*  end of: the_daemon  */


void init_daemon( void )
{
char *strptr;
struct sigaction sigact;
time_t lasttime = 0;

	sleep(3);
	ignore_all_signals(); 

	if( get_semaphore( DAEMON_SEMA ) == BAD )
	{
		dbugger(MANDATORY, "Can't get this programs semaphore\n\n");
		exit(BAD);
	}
	dbugger(MANDATORY, "Program: %s is INITIALIZING", program_name);

	sigact.sa_handler = sigusr;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;

	if (sigaction(SIGTERM, &sigact, (void *) NULL))
		dbugger(MANDATORY, "Sigaction - SIGTERM failed");
	if (sigaction(SIGUSR1, &sigact, (void *) NULL))
		dbugger(MANDATORY, "Sigaction - SIGUSR1 failed");

	open_pipe(pipe_name, Pipe_Message_Length);
	set_bit_dirty();

	dbugger(MANDATORY, "Program: %s is running in the backround", program_name);

} /* end of init_daemon */


void shutdown_daemon(void)
{
	close_pipe();
	dbugger(MANDATORY, "Pipe %s being deleted", pipe_name);
	del_semaphore(DAEMON_SEMA);
	dbugger(MANDATORY, "Semaphore %d being deleted", DAEMON_SEMA);
	set_bit_clean();
	dbugger(MANDATORY, "Program: %s is TERMINATING", program_name);

} /* end of shutdown_daemon */


void process_message(char *Message)
{
	switch(Message[0])
	{		case '0' :
				msg_handler_zero(Message); /* msgs starting with '0' are used */
				break;
			default :
				dbugger(IMPORTANT, "Invalid message received: %s",  Message);
				break;
	}
} /* end process_message */


void sigusr(int sig)
{
	if (sig == SIGTERM)
	{
		dbugger(INFORMATION, "*** Terminate signal received ***");
		Ready_Flag = FALSE;
	}
	if (sig == SIGUSR1)
	{
		dbugger(INFORMATION, "*** Reset signal received ***");
		Reset_Flag = TRUE;
	}
	if (sig == SIGALRM)
		dbugger(INFORMATION, "** An alarm has been raised **");

	dbugger(LOW_LEVEL, "** Signal %d has been raised **", sig);

} /* end sigusr */


int msg_handler_zero( char * new_message )
{
int err_code = 0;

	/* you might consider doing something with the messages being passed */

	dbugger(LOW_LEVEL, "Received from pipe: %s", new_message);

return err_code;
} /* end of msg_handler_zero */


/* **************************************** */
/* **************************************** */
/* ****  Supporting fifo-pipe functions *** */
/* **************************************** */
/* **************************************** */



void open_pipe( char* pipe_name_in, int pipe_message_size )
{	
	unlink(pipe_name_in);	
	
	if (mkfifo(pipe_name_in, FIFO_MODE) < 0)
	{
		dbugger(CRITICAL, "Fifo file %s already exists.", pipe_name_in);
		if (verify_pipe(pipe_name_in) == BAD) 
			exit(BAD);
	}

	if (chmod(pipe_name_in, S_IRWXU | S_IRWXG | S_IRWXO))
		dbugger(IMPORTANT, "Fifo file chmod failed");

	if ((Pipe_Handle = open(pipe_name_in, O_RDONLY | O_NONBLOCK)) < 0)
	{
		dbugger(CRITICAL, "Reading %s open failed.", pipe_name_in);
		exit(BAD);
	}
}



int verify_pipe(char * pipe_name)
{
struct stat filestat;

	if (stat(pipe_name, &filestat) < 0)
	{
		dbugger(CRITICAL, "Stat for %s failed.", pipe_name);
		return(BAD);
	}

	if (!S_ISFIFO(filestat.st_mode))
	{
		dbugger(CRITICAL, "%s filestat is %d, not a pipe.",
			pipe_name, filestat.st_mode);
		return(BAD);
	}

return(GOOD);
}  



int read_pipe( char* message_out )
{
int i;	
	if ((i = read(Pipe_Handle, message_out, Pipe_Message_Length)) > 0)
	{
		dbugger(INFORMATION, "Read %d bytes from pipe", i);
		return 1;
	}
return 0;
}


void close_pipe( void )
{	
	close( Pipe_Handle );	
	unlink( pipe_name );
}


/* **************************************** */
/* **************************************** */
/* ****  Supporting semaphore functions *** */
/* **************************************** */
/* **************************************** */


int check_semaphore( int semaphore )
{
int i;
		
	if ((i = semget((key_t) semaphore, 1, 0666 | SEM_UNDO)) == BAD)
	{
		dbugger(IMPORTANT, "Check semaphore fails: %d not found", semaphore);
		return(BAD);
	}
return(GOOD);
}


int get_semaphore(int semaphore){
	if ((sem_id = semget((key_t) semaphore, 1, 0666 | IPC_CREAT | SEM_UNDO)) == BAD)
	{
		dbugger(IMPORTANT, "Creating semaphore key %d failed", semaphore);
		return(BAD);
	}

	if(  set_semaphore(semaphore) == BAD)
	{
		dbugger(IMPORTANT, "Creating semaphore key %d failed", semaphore);
		return(BAD);
	}
	dbugger(IMPORTANT, "Creating semaphore key %d successful", semaphore);

return(GOOD);
} 



void del_semaphore(int semaphore)
{
union semun sem_union;
		
    if (semctl(sem_id, 0, IPC_RMID, sem_union) == -1)
	dbugger(LOW_LEVEL, "Releasing semaphore %d NOT successful", semaphore);
    else
	dbugger(LOW_LEVEL, "Releasing semaphore %d successful", semaphore);
}



int set_semaphore(int semaphore)
{
union semun sem_union;

    sem_union.val = 1;
    if (semctl(sem_id, 0, SETVAL, sem_union) == -1) 
    {
	dbugger(LOW_LEVEL, "Setting semaphore %d NOT successful", semaphore);
        return(BAD);
    }
    dbugger(LOW_LEVEL, "Setting semaphore %d successful", semaphore);

return(GOOD);
}


/* **************************************** */
/* ******* End of daemon.c **************** */
/* **************************************** */


int check_dirty_bit()
{
int is_dirty = FALSE;
char STAT_BIT[2];
int File_Handle;

	if ((File_Handle = open(INTEGRITY_FILE, O_RDWR | O_CREAT)) < 0)
	{
		fprintf(stderr, "Opening %s failed.\n", INTEGRITY_FILE);
		exit(BAD);
	}
	if (chmod(INTEGRITY_FILE, S_IRWXU | S_IRWXG | S_IRWXO))
	{
		fprintf(stderr, "Fifo file chmod failed\n");
		exit(BAD);
	}
	read(File_Handle, STAT_BIT, 1);

	if( STAT_BIT[0] == 'X' ) is_dirty = TRUE;
	if( STAT_BIT[0] == 'Z' ) is_dirty = FALSE;
	dbugger(TRIVIAL, "Char: [%d] - [%c] found in %s", 
		STAT_BIT[0], 
		STAT_BIT[0], 
		INTEGRITY_FILE);
	
	if( STAT_BIT[0] != 'X' && STAT_BIT[0] != 'Z' )
	{
		write( File_Handle, (const void *) "Z", 1 );
		dbugger(CRITICAL, 
			"Modified %s to have a Integrity bit", 
			INTEGRITY_FILE);
	}
	close(File_Handle);

return is_dirty;
}



int set_bit_clean()
{
int routine_complete = 0;
int File_Handle;

	if ((File_Handle = open(INTEGRITY_FILE, O_RDWR | O_CREAT)) < 0)
	{
		dbugger(IMPORTANT, "Opening %s failed.", INTEGRITY_FILE);
		exit(BAD);
	}
		write( File_Handle, (const void *) "Z", 1 );
		dbugger(INFORMATION, 
			"Modified %s to have a CLEAN Integrity bit", 
			INTEGRITY_FILE);
	close(File_Handle);

return routine_complete ;
}



int set_bit_dirty()
{
int routine_complete = 0;
int File_Handle;

	if ((File_Handle = open(INTEGRITY_FILE, O_RDWR | O_CREAT)) < 0)
	{
		dbugger(IMPORTANT, "Opening %s failed.", INTEGRITY_FILE);
		exit(BAD);
	}
		write( File_Handle, (const void *) "X", 1 );
		dbugger(INFORMATION, 
			"Modified %s to have a DIRTY Integrity bit", 
			INTEGRITY_FILE);
	close(File_Handle);

return routine_complete ;
}



int system_dirty()
{
int still_dirty = 1;

	/* Currently this function isn't filled. In the Real World */
	/* There would be system integrity checks..                */
	/* i.e. Database cleanup, flush temp files, fdisk, ...     */

	fprintf(stderr,"%s Recovered\n", program_name);
	still_dirty = 0;

return still_dirty;
}
