/////////////////////////////////////////////////////////////////////////////
//
//	Linked List demo II
//
//	Author: Chad Knudson
//
//	History:
//
//		05/01/97	Created. [ChadK]
//
/////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>

typedef struct _node 
{
	int iValue;
	struct _node * pNodeNext;
} NODE;

typedef struct _linkedlist
{
	NODE *	pNodeHead;
	int		cElements;
} LinkedList;

// Add bool type if needed
#ifndef BOOL
#define BOOL	int
#define TRUE	1
#define FALSE	0
#endif

#define NumberOfTestValues	6

/////////////////////////////////////////////////////////////////////////////
//							Function Prototypes
/////////////////////////////////////////////////////////////////////////////
BOOL FAddLinkedListElementAscending(LinkedList *, int);
BOOL FAddLinkedListElementDescending(LinkedList *, int);
BOOL FAddLinkedListElementHead(LinkedList *, int);
BOOL FAddLinkedListElementTail(LinkedList *, int);
void PrintListElements(LinkedList *);
void RemoveAllElementsFromList(LinkedList *);

/////////////////////////////////////////////////////////////////////////////
//								Main Program
/////////////////////////////////////////////////////////////////////////////
int main (int argc, char ** argv) 
{
	LinkedList ListHead;
	LinkedList ListTail;
	LinkedList ListSorted;
	LinkedList ListSortedRev;

	int iTestValuesArray[NumberOfTestValues] = { 10, 20, 30, 5, 40, 25 };
	int iTestValuesIndex;
	
	/////////////////////////////////////////////////////////////////////////
	// Initialize List member fields
	/////////////////////////////////////////////////////////////////////////
	ListHead.pNodeHead = NULL;
	ListHead.cElements = 0;
	ListTail.pNodeHead = NULL;
	ListTail.cElements = 0;
	ListSorted.pNodeHead = NULL;
	ListSorted.cElements = 0;
	ListSortedRev.pNodeHead = NULL;
	ListSortedRev.cElements = 0;

	/////////////////////////////////////////////////////////////////////////
	// Add some values to the list
	/////////////////////////////////////////////////////////////////////////
	for (iTestValuesIndex = 0; iTestValuesIndex < NumberOfTestValues; iTestValuesIndex++)
	{
		if (!FAddLinkedListElementHead(&ListHead, iTestValuesArray[iTestValuesIndex]))
		{
			fprintf(stderr, "Error adding a node to the linked list\n");
			exit(1);
		}
		if (!FAddLinkedListElementTail(&ListTail, iTestValuesArray[iTestValuesIndex]))
		{
			fprintf(stderr, "Error adding a node to the linked list\n");
			exit(1);
		}
		if (!FAddLinkedListElementAscending(&ListSorted, iTestValuesArray[iTestValuesIndex]))
		{
			fprintf(stderr, "Error adding a node to the linked list\n");
			exit(1);
		}
		if (!FAddLinkedListElementDescending(&ListSortedRev, iTestValuesArray[iTestValuesIndex]))
		{
			fprintf(stderr, "Error adding a node to the linked list\n");
			exit(1);
		}
	}

	/////////////////////////////////////////////////////////////////////////
	// Print out the values in the test data array
	/////////////////////////////////////////////////////////////////////////
	printf("The elements in the test data array are:\n");
	for (iTestValuesIndex = 0; iTestValuesIndex < NumberOfTestValues - 1; iTestValuesIndex++)
		printf("%d, ", iTestValuesArray[iTestValuesIndex]);
	printf("%d. ", iTestValuesArray[NumberOfTestValues - 1]);

	/////////////////////////////////////////////////////////////////////////
	// Print out the values in the list
	/////////////////////////////////////////////////////////////////////////
	printf("\n\nList with elements inserted at the head.\n");
	PrintListElements(&ListHead);

	printf("\n\nList with elements inserted at the tail.\n");
	PrintListElements(&ListTail);

	printf("\n\nList with elements inserted in order.\n");
	PrintListElements(&ListSorted);

	printf("\n\nList with elements inserted in reverse order.\n");
	PrintListElements(&ListSortedRev);

	/////////////////////////////////////////////////////////////////////////
	// Do our housekeeping -- free up all of the dynamically allocated memory
	/////////////////////////////////////////////////////////////////////////
	RemoveAllElementsFromList(&ListHead);
	RemoveAllElementsFromList(&ListTail);
	RemoveAllElementsFromList(&ListSorted);
	RemoveAllElementsFromList(&ListSortedRev);
 	
	return(0);
}

/////////////////////////////////////////////////////////////////////////////
//
//	FAddLinkedListElementHead
//
//	Purpose:	Add an element to the head of a list with an integer value.
//
//	Method:		Adds a node to the head of the list.
//
//	Returns:	TRUE  on success,
//				FALSE on failure.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
BOOL FAddLinkedListElementHead(LinkedList * pList, int iValue)
{
	NODE * pNodeNew;
	
	/////////////////////////////////////////////////////////////////////////
	// Create new node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew = (NODE *)malloc(sizeof(NODE));

	if (pNodeNew == NULL)
	   return(FALSE);

	/////////////////////////////////////////////////////////////////////////
	// Initialize the new node's pNodeNext to NULL and place iValue in the
	// node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew->pNodeNext = NULL;
	pNodeNew->iValue	= iValue;
	
	/////////////////////////////////////////////////////////////////////////
	// When we get here, pNodePrevious points to the node right before the
	// place where the new node should go.
	/////////////////////////////////////////////////////////////////////////
	if (pList->pNodeHead != NULL)
	{
		/////////////////////////////////////////////////////////////////////
		// Fixup the links with the previous node and the new node
		/////////////////////////////////////////////////////////////////////
		pNodeNew->pNodeNext = pList->pNodeHead;
	}
	pList->pNodeHead = pNodeNew;

	pList->cElements++;	// Increase our count of elements in the list

	return(TRUE);
	
}

/////////////////////////////////////////////////////////////////////////////
//
//	FAddLinkedListElementAscending
//
//	Purpose:	Add an element to the list with an integer value.
//
//	Method:		Adds a node to the list such that the list remains sorted
//				in ascending order.
//
//	Returns:	TRUE  on success,
//				FALSE on failure.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
BOOL FAddLinkedListElementAscending(LinkedList * pList, int iValue)
{
	NODE * pNodeCurrent;
	NODE * pNodeNew;
	NODE * pNodePrevious;
	
	/////////////////////////////////////////////////////////////////////////
	// Create new node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew = (NODE *)malloc(sizeof(NODE));

	if (pNodeNew == NULL)
	   return(FALSE);

	/////////////////////////////////////////////////////////////////////////
	// Initialize the new node's pNodeNext to NULL and place iValue in the
	// node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew->pNodeNext = NULL;
	pNodeNew->iValue	= iValue;
	
	/////////////////////////////////////////////////////////////////////////
	// Get things ready to traverse the list
	/////////////////////////////////////////////////////////////////////////
	pNodePrevious		= NULL;
	pNodeCurrent		= pList->pNodeHead;
	
	/////////////////////////////////////////////////////////////////////////
	// Traverse the list -- we want the new node to be in its proper place
	// such that the list remains sorted in ascending order.
	/////////////////////////////////////////////////////////////////////////
	while ((pNodeCurrent != NULL) && (iValue > pNodeCurrent->iValue))
	{
		pNodePrevious	= pNodeCurrent;
		pNodeCurrent	= pNodeCurrent->pNodeNext;
	}

	/////////////////////////////////////////////////////////////////////////
	// When we get here, pNodePrevious points to the node right before the
	// place where the new node should go.
	/////////////////////////////////////////////////////////////////////////
	if (pNodePrevious != NULL)
	{
		/////////////////////////////////////////////////////////////////////
		// Fixup the links with the previous node and the new node
		/////////////////////////////////////////////////////////////////////
		pNodeNew->pNodeNext = pNodePrevious->pNodeNext;
		pNodePrevious->pNodeNext = pNodeNew;
	}
	else
	{
		/////////////////////////////////////////////////////////////////////
		// This must be the first node of the list
		/////////////////////////////////////////////////////////////////////
		pNodeNew->pNodeNext = pList->pNodeHead;
		pList->pNodeHead = pNodeNew;

		/////////////////////////////////////////////////////////////////////
		// pNodeNew->pNodeNext was already initialized to NULL so we don't 
		// have to do anything with it here
		/////////////////////////////////////////////////////////////////////
	}

	pList->cElements++;	// Increase our count of elements in the list

	return(TRUE);
	
}

/////////////////////////////////////////////////////////////////////////////
//
//	FAddLinkedListElementDescending
//
//	Purpose:	Add an element to the list with an integer value.
//
//	Method:		Adds a node to the list such that the list remains sorted
//				in descending order.
//
//	Returns:	TRUE  on success,
//				FALSE on failure.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
BOOL FAddLinkedListElementDescending(LinkedList * pList, int iValue)
{
	NODE * pNodeCurrent;
	NODE * pNodeNew;
	NODE * pNodePrevious;
	
	/////////////////////////////////////////////////////////////////////////
	// Create new node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew = (NODE *)malloc(sizeof(NODE));

	if (pNodeNew == NULL)
	   return(FALSE);

	/////////////////////////////////////////////////////////////////////////
	// Initialize the new node's pNodeNext to NULL and place iValue in the
	// node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew->pNodeNext = NULL;
	pNodeNew->iValue	= iValue;
	
	/////////////////////////////////////////////////////////////////////////
	// Get things ready to traverse the list
	/////////////////////////////////////////////////////////////////////////
	pNodePrevious		= NULL;
	pNodeCurrent		= pList->pNodeHead;
	
	/////////////////////////////////////////////////////////////////////////
	// Traverse the list -- we want the new node to be in its proper place
	// such that the list remains sorted in descending order.
	/////////////////////////////////////////////////////////////////////////
	while ((pNodeCurrent != NULL) && (iValue < pNodeCurrent->iValue))
	{
		pNodePrevious	= pNodeCurrent;
		pNodeCurrent	= pNodeCurrent->pNodeNext;
	}

	/////////////////////////////////////////////////////////////////////////
	// When we get here, pNodePrevious points to the node right before the
	// place where the new node should go.
	/////////////////////////////////////////////////////////////////////////
	if (pNodePrevious != NULL)
	{
		/////////////////////////////////////////////////////////////////////
		// Fixup the links with the previous node and the new node
		/////////////////////////////////////////////////////////////////////
		pNodeNew->pNodeNext = pNodePrevious->pNodeNext;
		pNodePrevious->pNodeNext = pNodeNew;
	}
	else
	{
		/////////////////////////////////////////////////////////////////////
		// This must be the first node of the list
		/////////////////////////////////////////////////////////////////////
		pNodeNew->pNodeNext = pList->pNodeHead;
		pList->pNodeHead = pNodeNew;

		/////////////////////////////////////////////////////////////////////
		// pNodeNew->pNodeNext was already initialized to NULL so we don't 
		// have to do anything with it here
		/////////////////////////////////////////////////////////////////////
	}

	pList->cElements++;	// Increase our count of elements in the list

	return(TRUE);
	
}

/////////////////////////////////////////////////////////////////////////////
//
//	FAddLinkedListElementTail
//
//	Purpose:	Add an element to the end of the list with an integer value.
//
//	Method:		Adds a node to the end of the list.
//
//	Returns:	TRUE  on success,
//				FALSE on failure.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
BOOL FAddLinkedListElementTail(LinkedList * pList , int iValue)
{
	NODE * pNodeCurrent;
	NODE * pNodeNew;
	NODE * pNodePrevious;
	
	/////////////////////////////////////////////////////////////////////////
	// Create new node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew = (NODE *)malloc(sizeof(NODE));

	if (pNodeNew == NULL)
	   return(FALSE);

	/////////////////////////////////////////////////////////////////////////
	// Initialize the new node's pNodeNext to NULL and place iValue in the
	// node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew->pNodeNext = NULL;
	pNodeNew->iValue    = iValue;
	
	/////////////////////////////////////////////////////////////////////////
	// Get things ready to traverse the list
	/////////////////////////////////////////////////////////////////////////
	pNodePrevious		= NULL;
	pNodeCurrent		= pList->pNodeHead;
	
	/////////////////////////////////////////////////////////////////////////
	// Traverse the list -- we want the new node to be at the end of the list
	/////////////////////////////////////////////////////////////////////////
	while (pNodeCurrent != NULL)
	{
		pNodePrevious	= pNodeCurrent;
		pNodeCurrent	= pNodeCurrent->pNodeNext;
	}

	/////////////////////////////////////////////////////////////////////////
	// When we get here, pNodePrevious points to the node right before the
	// place where the new node should go.
	/////////////////////////////////////////////////////////////////////////
	if (pNodePrevious != NULL)
	{
		/////////////////////////////////////////////////////////////////////
		// Fixup the links with the previous node and the new node
		/////////////////////////////////////////////////////////////////////
		pNodeNew->pNodeNext = pNodePrevious->pNodeNext;
		pNodePrevious->pNodeNext = pNodeNew;
	}
	else
	{
		/////////////////////////////////////////////////////////////////////
		// This must be the first node of the list
		/////////////////////////////////////////////////////////////////////
		pList->pNodeHead = pNodeNew;

		/////////////////////////////////////////////////////////////////////
		// pNodeNew->pNodeNext was already initialized to NULL so we don't 
		// have to do anything with it here
		/////////////////////////////////////////////////////////////////////
	}

	pList->cElements++;	// Increase our count of elements in the list

	return(TRUE);
	
}

/////////////////////////////////////////////////////////////////////////////
//
//	PrintListElements
//
//	Purpose:	Print the values in the nodes of an integer list.
//
//	Method:		
//
//	Returns:	Nothing.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
void PrintListElements(LinkedList * pList)
{
	NODE *	pNodeCurrent;
	int		iElement = 0;

	pNodeCurrent = pList->pNodeHead;
	
	printf("The linked list has %d nodes.\n\n", pList->cElements);

	/////////////////////////////////////////////////////////////////////////
	// Traverse the list -- we want the new node to be at the end of the list
	/////////////////////////////////////////////////////////////////////////
	while (pNodeCurrent != NULL)
	{
		printf("%d", pNodeCurrent->iValue);
		
		if (++iElement < pList->cElements)
		{
			printf(", ");
		}
		else
		{
			printf(".\n");
		}

		pNodeCurrent = pNodeCurrent->pNodeNext;
	}
}

/////////////////////////////////////////////////////////////////////////////
//
//	RemoveAllElementsFromList
//
//	Purpose:	Free the dynamic memory allocated for a list.
//
//	Method:		
//
//	Returns:	Nothing.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
void RemoveAllElementsFromList(LinkedList * pList)
{
	NODE *	pNodeKill;
	NODE *	pNodeCurrent;

	pNodeKill	 = NULL;
	pNodeCurrent = pList->pNodeHead;
	
	/////////////////////////////////////////////////////////////////////////
	// Traverse the list -- we want to free each node's memory
	/////////////////////////////////////////////////////////////////////////
	while (pNodeCurrent != NULL)
	{
		pNodeKill		= pNodeCurrent;
		pNodeCurrent	= pNodeCurrent->pNodeNext;
		free(pNodeKill);
	}

	/////////////////////////////////////////////////////////////////////////
	// Re-initialize List member fields
	/////////////////////////////////////////////////////////////////////////
	pList->pNodeHead = NULL;
	pList->cElements = 0;

}